/*
 *   Copyright 2000 Michael J. Roberts.
 *   
 *   This file is part of TADS 3. 
 */


/* ------------------------------------------------------------------------ */
/*
 *   TADS datatype codes.  These values are returned by propType(), etc.  
 */
#define TYPE_NIL         1
#define TYPE_TRUE        2
#define TYPE_OBJECT      5
#define TYPE_PROP        6
#define TYPE_INT         7
#define TYPE_SSTRING     8
#define TYPE_DSTRING     9
#define TYPE_LIST        10
#define TYPE_CODE        11
#define TYPE_FUNCPTR     12
#define TYPE_NATIVE_CODE 14
#define TYPE_ENUM        15


/* ------------------------------------------------------------------------ */
/*
 *   The root object class.  All objects descend from this class. 
 */
intrinsic class Object 'root-object/030001'
{
    /* determine if I'm an instance of the given class 'cls' */
    isClass(cls);

    /* get the list of direct superclasses of this object */
    getSuperclassList();

    /* determine if a property is defined for this object */
    propDefined(prop, flags?);

    /* get the type of a property defined for this object */
    propType(prop);
}

/*
 *   propDefined() flags 
 */
#define PROPDEF_ANY           1
#define PROPDEF_DIRECTLY      2
#define PROPDEF_INHERITS      3
#define PROPDEF_GET_CLASS     4


/* ------------------------------------------------------------------------ */
/*
 *   The IntrinsicClass intrinsic class 
 */
intrinsic class IntrinsicClass 'intrinsic-class/030000'
{
}


/* ------------------------------------------------------------------------ */
/*
 *   Anonymous function pointer intrinsic class 
 */
intrinsic class AnonFuncPtr 'anon-func-ptr'
{
}


/* ------------------------------------------------------------------------ */
/*
 *   The "TADS Object" intrinsic class.  All objects that the program
 *   defines with the "class" or "object" statements descend from this
 *   class.  
 */
intrinsic class TadsObject 'tads-object/030000'
{
}


/* ------------------------------------------------------------------------ */
/*
 *   The native string type. 
 */
intrinsic class String 'string/030001'
{
    /* get the length of the string */
    length();

    /* extract a substring */
    substr(start, len?);

    /* convert to upper case */
    toUpper();

    /* convert to lower case */
    toLower();

    /* find a substring */
    find(str);

    /* 
     *   convert to a list of Unicode character codes, or get the Unicode
     *   character code for the single character at the given index 
     */
    toUnicode(idx?);

    /* htmlify a string */
    htmlify(flags?);
}

/*
 *   flags for String.htmlify 
 */

/* 
 *   preserve spaces - converts each space in a run of multiple spaces to
 *   a &nbsp; sequence 
 */
#define STR_HTML_KEEP_SPACES     0x0001

/* preserve newlines - converts each \n character to a <br> tag */
#define STR_HTML_KEEP_NEWLINES   0x0002

/* preserve tabs - converts each \t character to a <tab> tag */
#define STR_HTML_KEEP_TABS       0x0004

/* 
 *   preserve all whitespace exactly as is - preserves all spaces, tabs,
 *   and newlines 
 */
#define STR_HTML_KEEP_WHITESPACE \
    (STR_HTML_KEEP_SPACES | STR_HTML_KEEP_NEWLINES | STR_HTML_KEEP_TABS)


/* ------------------------------------------------------------------------ */
/*
 *   The native collection type - this is the base class for lists,
 *   arrays, and other objects that represent collections of values. 
 */
intrinsic class Collection 'collection/030000'
{
    /* 
     *   Create an iterator for the collection.  This returns a new
     *   Iterator object that can be used to iterate over the values in
     *   the collection.  The Iterator will use a snapshot of the
     *   collection that will never change, even if the collection is
     *   changed after the iterator is created.  
     */
    createIterator();

    /*
     *   Create a "live iterator" for the collection.  This returns a new
     *   Iterator object that refers directly to the original collection;
     *   if the original collection changes, the iterator will reflect the
     *   changes in its iteration.  As a result, the iterator is not
     *   guaranteed to visit all of the elements in the collection if the
     *   collection changes during the course of the iteration.  If
     *   consistent results are required, use createIterator() instead.  
     */
    createLiveIterator();
}

/* ------------------------------------------------------------------------ */
/*
 *   The native iterator type - this is the base class for all iterators.
 *   This class is abstract and is thus never directly instantiated.
 *   
 *   Note that iterators can never be created directly with the 'new'
 *   operator.  Instead, iterators must be obtained from a collection via
 *   the collection's createIterator() method.  
 */
intrinsic class Iterator 'iterator/030000'
{
    /*
     *   Get the next item in the collection.  This returns the next item,
     *   and advances the internal state in the iterator so that a
     *   subsequent call to getNext() returns the next item after this
     *   one.  When the iterator is first created, or after calling
     *   resetIterator(), this returns the first item in the collection.  
     */
    getNext();

    /*
     *   Determine if the collection is out of items.  Returns true if
     *   getNext() will return a valid item, nil if no more items are
     *   available.  
     */
    isNextAvailable();

    /*
     *   Reset to the first item.  After calling this routine, the next
     *   call to getNext() will return the first item in the collection. 
     */
    resetIterator();
}

/*
 *   Indexed object iterator - this type of iterator is used for lists,
 *   arrays, and other indexed collection objects.  
 */
intrinsic class IndexedIterator 'indexed-iterator/030000'
{
};


/* ------------------------------------------------------------------------ */
/*
 *   The native list type 
 */
intrinsic class List 'list/030005'
{
    /* 
     *   Select a subset of the list: returns a new list consisting only
     *   of the elements for which the callback function 'func' returns
     *   true.  
     */
    subset(func);

    /*
     *   Apply the callback function to each element of this list, and
     *   return a new list consisting of the results.  Effectively maps
     *   the list to a new list using the given function.  Suppose the
     *   original list is
     *   
     *   [x, y, z]
     *   
     *   Then the result list is
     *   
     *   [func(x), func(y), func(z)] 
     */
    mapAll(func);

    /* get the number of elements in the list */
    length();

    /* extract a sublist */
    sublist(start, len?);

    /* intersect with another list */
    intersect(other);

    /* get the index of the first match for the given value */
    indexOf(val);

    /* car/cdr - head/tail of list */
    car();
    cdr();

    /* 
     *   Find the first element for which the given condition is true, and
     *   return the index of the element.  Applies the callback function
     *   (which encodes the condition to evaluate) to each element in
     *   turn, starting with the first.  For each element, if the callback
     *   returns nil, proceeds to the next element; otherwise, stops and
     *   returns the index of the element.  If the callback never returns
     *   true for any element, we'll return nil.  
     */
    indexWhich(cond);

    /* 
     *   Invoke the callback on each element, in order from first to last.
     *   No return value. 
     */
    forEach(func);

    /* 
     *   Find the first element for which the given condition is true, and
     *   return the value of the element.  Returns nil if no item
     *   satisfies the condition.  
     */
    valWhich(cond);

    /* find the last element with the given value, and return its index */
    lastIndexOf(val);

    /* 
     *   Find the last element for which the condition is true, and return
     *   the index of the element.  Applies the callback to each element
     *   in turn, starting with the last element and working backwards.
     *   For each element, if the callback returns nil, proceeds to the
     *   previous element; otherwise, stops and returns the index of the
     *   element.  If the callback never returns true for any element,
     *   we'll return nil.  
     */
    lastIndexWhich(cond);

    /* 
     *   Find the last element for which the condition is true, and return
     *   the value of the element 
     */
    lastValWhich(cond);

    /* count the number of elements with the given value */
    countOf(val);

    /* count the number of elements for which the callback returns true */
    countWhich(cond);

    /* get a new list consisting of the unique elements of this list */
    getUnique();

    /* 
     *   append the elements of the list 'lst' to the elements of this
     *   list, then remove repeated elements in the result; returns a new
     *   list with the unique elements of the combination of the two lists 
     */
    appendUnique(lst);

    /* 
     *   append an element - this works almost exactly like the
     *   concatation operator ('+'), but if the argument is a list, this
     *   simply adds the list as a new element, rather than adding each
     *   element of the list as a separate element 
     */
    append(val);

    /* 
     *   Sort the list, returning a new list.  If the 'descending' flag is
     *   provided and is not nil, we'll sort the list in descending order
     *   rather than ascending order.
     *   
     *   If the 'comparisonFunction' argument is provided, it must be a
     *   callback function; the callback takes two arguments, and returns
     *   an integer less than zero if the first argument value is less
     *   than the second, zero if they're equal, and an integer greater
     *   than zero if the first is greater than the second.  If no
     *   'comparisonFunction' argument is provided, or it's provided and
     *   its value is nil, we'll simply compare the list elements as
     *   ordinary values.  The comparison function can be provided for
     *   caller-defined orderings, such as ordering a set of objects.  
     */
    sort(descending?, comparisonFunction?);
}

