/* Copyright (c) 2002 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  osgen.h - definitions for the osgen subsystem
Function
  
Notes
  
Modified
  05/19/02 MJRoberts  - Creation
*/

#ifndef OSGEN_H
#define OSGEN_H

/* ------------------------------------------------------------------------ */
/*
 *   Text attributes
 */

/* 
 *   BOLD - use highlighted text.  On most character-mode platforms, this
 *   will map to a bright color or other emphasized status.  Platforms are
 *   free to use any desired rendering, but if possible the rendering should
 *   look different from plain text, and should connote some sort of
 *   emphasis. 
 */
#define OSGEN_ATTR_BOLD     0x01

/* ------------------------------------------------------------------------ */
/*
 *   osgen support routines.  These routines are implemented by the
 *   semi-portable osgen.c layer for use by the underlying osxxx code.  
 */

/*
 *   Initialize the osgen scrollback buffer.  Allocate the given amount of
 *   space in bytes for scrollback.
 */
void osssbini(unsigned int size);

/*
 *   Delete the scrollback buffer.  This should be called during program
 *   termination to release memory allocated by osssbini().
 */
void osssbdel(void);



/* ------------------------------------------------------------------------ */
/*
 *   ossxxx implementation for osgen.  These routines must be defined by the
 *   OS-specific subsystem. 
 */

/*
 *   Get extended system information.  This checks to see if the ossxxx
 *   implementation handles the system information code.  If the osxxx layer
 *   doesn't want to handle the code, it can return false so that the osgen
 *   layer will provide a default interpretation.
 *   
 *   Currently, the following codes should be handled by the oss layer:
 *   
 *   SYSINFO_TEXT_COLORS - return the level of color support on this
 *   platform.  Because the ossxxx color interface that osgen uses is limited
 *   to the ANSI colors, this should normally return one of the ANSI support
 *   levels if color is supported, or the unsupported level if color isn't
 *   supported.  Note that if the platform uses a fixed color scheme that
 *   cannot be controlled via the ossxxx routines, this should usually return
 *   the "highlighting only" or "parameterized colors" level.
 *   
 *   SYSINFO_TEXT_HILITE - indicate whether or not highlighted text is
 *   rendered in a distinctive appearance.  
 */
int oss_get_sysinfo(int code, void *param, long *result);


/*
 *   Translate a portable color specification to an oss-style color code.
 *   This returns a color code suitable for use in ossclr(), ossdsp(), and
 *   the other ossxxx() routines defined here that take color codes as
 *   parameters; this color code is opaque to the caller.
 *   
 *   The purpose of this routine is to translate from the portable interface
 *   to color to the hardware-specific or OS-specific color coding used in
 *   the ossxxx layer, so that a caller given a color specified via the
 *   portable interface can send the appropriate ossxxx color code to the
 *   lower-level routines.
 *   
 *   'fg', 'bg', and 'screen_color' are OS_COLOR_xxx values.  'attrs' is a
 *   combination of OSGEN_ATTR_xxx values.  This routine should combine the
 *   colors specified by the combination of these values to yield an
 *   appropriate color code for use in the ossxxx routines.
 *   
 *   Note that the 'bg' color can be OS_COLOR_TRANSPARENT.  If this is the
 *   case, the 'screen_color' value should be used to determine the
 *   background color, if possible.  'fg' should never be transparent.  If
 *   'bg' is not transparent, then screen color is usually ignored.
 *   
 *   COLOR SUPPORT IS OPTIONAL, but this routine must be implemented if
 *   osgen.c is to be used.  If the platform does NOT support explicit color
 *   settings, this routine should still return an appropriate code that can
 *   be used with ossclr, etc.  For example, the Unix oss implementation, at
 *   the time of this writing, used an internal color scheme to select from a
 *   fixed set of parameterized colors (normal, highlighted, reverse), so the
 *   Unix implementation could simply return its appropriate internal code,
 *   largely ignoring the requested colors.  Note, however, that a platform
 *   with only parameterized colors might still want to inspect the 'fg' and
 *   'bg' colors for the OS_COLOR_xxx parameterized colors (OS_COLOR_TEXT,
 *   OS_COLOR_STATUSBG, etc), and return its internal code corresponding to
 *   the selected parameter color when possible.  
 */
int ossgetcolor(int fg, int bg, int attrs, int screen_color);

/*
 *   Clear an area of the screen.  The color code is opaque to callers; it is
 *   meaningful only to the ossxxx implementation.  
 */
void ossclr(int top, int left, int bottom, int right, int color);

/*
 *   Display text at a particular location in a particular color.
 */
void ossdsp(int line, int column, int color, const char *msg);

/*
 *   Move the cursor 
 */
void ossloc(int line, int column);

/*
 *   Scroll a region of the screen UP one line. 
 */
void ossscu(int top_line, int left_column, int bottom_line,
            int right_column, int blank_color);

/*
 *   Scroll a region of the screen DOWN one line. 
 */
void ossscr(int top_line, int left_column, int bottom_line,
            int right_column, int blank_color);



#endif /* OSGEN_H */
