/* Copyright (c) 2002 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  osscolor.c - ossxxx color routines for DOS
Function
  
Notes
  
Modified
  05/19/02 MJRoberts  - Creation
*/

#include "os.h"
#include "osgen.h"

extern int os_f_plain;

/*
 *   Given an OS_COLOR_xxx value, translate to the 3-bit system color:
 *   
 *.  0 = black
 *.  1 = blue
 *.  2 = green
 *.  3 = cyan
 *.  4 = red
 *.  5 = magenta
 *.  6 = yellow
 *.  7 = white
 *   
 *   If the color value is one of the "parameterized" colors, we'll use the
 *   current value for the color as set in the global variables
 *   (text_normal_color, etc).  
 */
static int xlat_color(int color)
{
    switch(color)
    {
    case OS_COLOR_TEXT:
        /* return the foreground portion of the normal text color */
        return (text_normal_color & 0x07);

    case OS_COLOR_TEXTBG:
        /* return the background portion of the normal text color */
        return ((text_normal_color >> 4) & 0x07);

    case OS_COLOR_STATUSLINE:
        /* return the foreground portion of the statusline color */
        return (sdesc_color & 0x07);

    case OS_COLOR_STATUSBG:
        /* return the background portion of the statusline color */
        return ((sdesc_color >> 4) & 0x07);

    case OS_COLOR_INPUT:
        /* return the foreground portion of the normal text color */
        return (text_normal_color & 0x07);

    case OS_COLOR_BLACK:
        return 0;

    case OS_COLOR_BLUE:
        return 1;

    case OS_COLOR_GREEN:
        return 2;

    case OS_COLOR_CYAN:
        return 3;

    case OS_COLOR_RED:
        return 4;

    case OS_COLOR_MAGENTA:
        return 5;

    case OS_COLOR_YELLOW:
        return 6;

    case OS_COLOR_WHITE:
        return 7;

    default:
        return 0;
    }
}

/*
 *   Translate a portable color specifiation to an ossxxx value.
 *   
 *   For DOS character mode, the OS-level color consists of a combination of
 *   bit fields:
 *   
 *   u b b b h f f f
 *   
 *   'u' = unused
 *.  'b' = background color - a 3-bit value in bits 5-7
 *.  'h' = highlight - a 1-bit value at bit 4
 *.  'f' = foreground color - a 3-bit value in bits 1-3
 *   
 *   The foreground and background values are taken from the set listed in
 *   the comments on xlat_color() above.
 */
int ossgetcolor(int fg, int bg, int attrs, int screen_color)
{
    int color;

    /*
     *   Check for a special case: if they're just setting the BOLD attribute
     *   for text in the normal text color (the PARAMETERIZED normal text
     *   color, not the color that happens to match the current parameter
     *   value), then use the text_bold_color global variable, which comes
     *   from the user preferences. 
     */
    if (fg == OS_COLOR_TEXT && bg == OS_COLOR_TRANSPARENT
        && (attrs & OSGEN_ATTR_BOLD) != 0)
    {
        /* 
         *   they specifically want bold normal text, so use the
         *   parameterized setting for this style 
         */
        return text_bold_color;
    }
    
    /*
     *   First, figure the background color.  If 'bg' is transparent, then
     *   use the current screen color.  Otherwise, use the given background
     *   color.  
     */
    if (bg == OS_COLOR_TRANSPARENT)
    {
        /* use the screen color */
        color = xlat_color(screen_color);
    }
    else
    {
        /* use the explicit color */
        color = xlat_color(bg);
    }

    /* shift the background color into the appropriate bit field */
    color <<= 4;

    /* combine the background color with the foreground color */
    color |= xlat_color(fg);

    /* if the BOLD attribute is set, set the highlight bit in the color */
    if ((attrs & OSGEN_ATTR_BOLD) != 0)
        color |= 0x08;

    /* return the color */
    return color;
}

/*
 *   Get extended system information 
 */
int oss_get_sysinfo(int code, void *param, long *result)
{
    switch(code)
    {
    case SYSINFO_TEXT_COLORS:
        /* we support the ANSI colors, foreground and background */
        *result = SYSINFO_TXC_ANSI_FGBG;
        return TRUE;

    case SYSINFO_TEXT_HILITE:
        /* 
         *   we can render highlighted text with a unique appearance, as long
         *   as we're not in 'plain' mode 
         */
        *result = (os_f_plain ? 0 : 1);
        return TRUE;

    default:
        /* not recognized */
        return FALSE;
    }
}

